from ultralytics import YOLO
import os
import glob
from datetime import datetime
from PIL import Image
import time
import schedule

model = YOLO(os.path.join('Models','model.pt'))



def convert_jpg_to_bmp(jpg_path, bmp_path):
    """
    Convert a JPG image to 24-bit RGB BMP format
    
    Args:
        jpg_path (str): Path to input JPG file
        bmp_path (str): Path to save output BMP file
    """
    try:
        # Open the JPG image
        with Image.open(jpg_path) as img:
            # Convert to RGB if not already (handles CMYK, grayscale, etc.)
            if img.mode != 'RGB':
                img = img.convert('RGB')
            
            # Save as 24-bit BMP
            img.save(bmp_path, 'BMP')
            
        print(f"Successfully converted {jpg_path} to {bmp_path}")
        return True
    
    except Exception as e:
        print(f"Conversion failed: {str(e)}")
        return False

class YOLOv8deploy:
    
    def __init__(self, model: YOLO, images_folder: str):
        
        """A simple class for deploying YOLOv8 model for detecting trading patterns in chart images

        Args:
            model (YOLO): YOLO model object
            images_folder (str): A path where images will be imported from
        """
        
        self.model = model
        self.images_folder = images_folder
        
    def _get_images(self, folder: str, img_extensions: list=['*.png', '*.jpg', '*.jpeg']) -> tuple: 
        
        """ A function to help us detect the number of images present in a folder

        Args:
            folder (str): A path where images are located
            img_extensions (list, optional): Image filenames extensions. Defaults to ['*.png', '*.jpg', '*.jpeg'].

        Returns:
            tuple: Returns the number of images present in a folder and their names
        """
        
        image_files = []
        for ext in img_extensions:
            image_files.extend(glob.glob(os.path.join(folder, ext)))

        return (len(image_files), image_files) # Get the number of images and their names
        
    def predict_image(self, img_name: str, save_path: str):
        
        """This function predicts a single image

        Args:
            img_name (str): name of the image
            hist (bool, optional): When set to false it means the function isn't predicting multiple instances and the outcome will be displayed.
            
            Defaults to True.
        """
        
        if os.path.exists(img_name) == False: # Check if an image exists
            print(f"Failed to detect patterns, {img_name} not found")
            return
        
        results = self.model.predict(source=img_name, 
                                     save=True,
                                     project=save_path,
                                     name="YOLOv8 Images",
                                     exist_ok=True
                                     ) # Predict an image 
                    
        # Loop through the results
        for result in results:
            boxes = result.boxes  # Contains bounding boxes and confidence
            names = result.names  # Class index to name mapping

            # Convert a jpg image to bmp suitable for MQL5 diplay purposes
            
            base_name = os.path.splitext(os.path.basename(img_name))[0] + ".jpg"
            saved_path = os.path.join(result.save_dir, base_name)        

            convert_jpg_to_bmp(saved_path, os.path.join(result.save_dir, os.path.splitext(os.path.basename(img_name))[0] + '.bmp'))

            if boxes is not None and len(boxes) > 0:
                for box in boxes:
                    cls_id = int(box.cls[0])  # class id
                    conf = box.conf[0].item()  # confidence score
                    label = names[cls_id]
                    
                    print(f"Detected: {label} (confidence: {conf:.2f})")
            else:
                print("No detections.")


    def predict_images(self):
        
        _, image_names = self._get_images(self.images_folder) # Get all images from a folder
        
        for image_name in image_names:
            self.predict_image(img_name=image_name)



files_path = r"C:\Users\Omega Joctan\AppData\Roaming\MetaQuotes\Terminal\F4F6C6D7A7155578A6DEA66D12B1D40D\MQL5\Files"
images_path = os.path.join(files_path, "Screenshots") # Change this for to the right path on your pc :)
symbol = "EURUSD"
timeframe = "PERIOD_H1"

def scheduledYOLOv8Run():
        
    now = datetime.now() # Get the current local date and time

    # Extract current day and hour

    date = now.day
    current_day = now.weekday()  # e.g., 'Wednesday'
    current_hour = now.strftime("%H")  # e.g., '14' for 2 PM in 24-hour format

    image_filename = os.path.join(images_path, f"{symbol}.{timeframe}.{date}.{current_day+1}.{current_hour}.png")

    pattern_detector = YOLOv8deploy(model=model, images_folder=images_path)

    pattern_detector.predict_image(img_name=image_filename, 
                                    save_path=files_path)

    print(f"Processed image at {datetime.now().strftime('%Y-%m-%d %H:%M:%S')}")
    


# Schedule the pattern detection after every hour

schedule.every(1).minutes.do(scheduledYOLOv8Run)

print("Scheduler started. Press Ctrl+C to stop.")

# Run forever
while True:
    schedule.run_pending()
    time.sleep(1)